<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Cart;
use App\Models\Invoice;
use App\Models\Notification;
use App\Models\Order;
use App\Models\RefundRequest;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class OrderController extends Controller
{
    public function store(Request $request)
    {
        try {

            $rules = [
                'cart_identifier' => 'required|string',
                'order_type' => 'required|string|in:delivery,pick_up',
                'payment_method' => 'required|string|in:cod,card',
            ];

            $validate = Validator::make($request->all(), $rules);

            if ($validate->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation Error.',
                    'data' => $validate->errors(),
                ], 422);
            }

            $user = auth()->user();


            $rules = [];

            if ($request->order_type == 'delivery') {
                $rules['shipping_address_id'] = 'required|exists:user_addresses,id,type,shipping,user_id,' . $user->id;
                $rules['billing_address_id'] = 'nullable|exists:user_addresses,id,type,billing';
                $rules['billing_address_same_as_shipping'] = 'nullable|boolean';

                if (!$request->billing_address_same_as_shipping) {
                    $rules['billing_address_id'] = 'required|exists:user_addresses,id,type,billing,user_id,' . $user->id;
                }
            } else {
                // Pickup address and billing address required
                $rules['pickup_address_id'] = 'required|exists:pickup_addresses,id';
                $rules['billing_address_id'] = 'required|exists:user_addresses,id,type,billing,user_id,' . $user->id;
            }



            $validate = Validator::make($request->all(), $rules);

            if ($validate->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation Error.',
                    'data' => $validate->errors(),
                ], 422);
            }


            $cart = Cart::where('cart_identifier', $request->cart_identifier)
                ->with('cart_items')
                ->first();

            if (!$cart) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart is empty.',
                ], 404);
            }

            if ($cart->user_id != $request->user_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart does not belong to the user.',
                ], 403);
            }

            if ($cart->cart_items->count() <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart is empty.',
                ], 404);
            }

            DB::beginTransaction();

            $total_amount = 0;
            foreach ($cart->cart_items as $cart_item) {
                $total_amount += $cart_item->price * $cart_item->quantity;
            }

            $grand_total = $total_amount;

            $include_vat_tax = get_setting('include_vat_tax');
            if ($include_vat_tax == '1') {
                $vat_tax_percentage = get_setting('vat_tax_percentage');
                $vat_tax_amount = ($total_amount * $vat_tax_percentage) / 100;
                $grand_total = $total_amount + $vat_tax_amount;
            }

            $order_data = [
                'shipping_address_id' => $request->shipping_address_id,
                'billing_address_id' => $request->billing_address_id,
                'same_as_shipping_address' => $request->billing_address_same_as_shipping ? 1 : 0,
                'pickup_address_id' => $request->pickup_address_id,
                'order_number' => uniqid(),
                'status' => 'pending',
                'payment_status' => 'pending',
                'payment_method' => $request->payment_method,
                'delivery_method' => $request->order_type,
                'vat_tax_percentage' => $vat_tax_percentage ?? null,
                'vat_tax_amount' => $vat_tax_amount ?? null,
                'total_amount' => $total_amount,
                'grand_total' => $grand_total,
            ];

            if ($request->order_type == 'pick_up') {
                $order_data['shipping_charges'] = 0;
            }

            $order = $user->orders()->create($order_data);

            foreach ($cart->cart_items as $cart_item) {
                $order->orderDetails()->create([
                    'product_id' => $cart_item->product_id,
                    'quantity' => $cart_item->quantity,
                    'price' => $cart_item->price,
                    'total_amount' => $cart_item->price * $cart_item->quantity,
                ]);
            }

            $order_log_data = [
                'action_by' => $user->id,
                'order_id' => $order->id,
                'status' => 'pending',
                'action' => 'order_placed',
                'description' => 'An order has been placed.',
            ];

            $order->orderLogs()->create($order_log_data);

            //send notification to admin
            $notification_data = [
                'message' => 'An order has been placed.',
                'user_id' => $user->id,
                'recepient_id' => getAdminId(),
                'type' => 'general',
                'url' => route('admin.order.details', ['id' => $order->id]),
                'avatar' => auth()->user()->avatar,
            ];

            Notification::create($notification_data);

            //manage stock
            foreach ($cart->cartItems as $cartItem) {
                $product = $cartItem->product;
                $product->quantity = $product->quantity - $cartItem->quantity;
                $product->save();
            }

            $cart->cartItems()->delete();
            $cart->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Order placed successfully.',
                'data' => [
                    'order' => $order->only('id', 'order_number', 'total_amount', 'grand_total', 'status', 'payment_status', 'payment_method', 'delivery_method', 'vat_tax_percentage', 'vat_tax_amount', 'shipping_charges', 'created_at', 'orderDetails'),
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong.',
                'data' => $e->getMessage(),
            ], 500);
        }
    }

    public function index(Request $request)
    {
       $validate = Validator::make($request->all(), [
        //'pending','approved','decline','packing','performa_invoice','invoice_review','delivered','cancelled', 'shipping', 'invoiced' ,'packed','shipped'
            'status' => 'nullable|in:pending,approved,decline,packing,performa_invoice,invoice_review,delivered,cancelled,shipping,invoiced,packed,shipped',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'page' => 'integer',
            'per_page' => 'integer',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $user = auth()->user();

        $orders = Order::with(['shippingAddress', 'billingAddress', 'pickupAddress'])
            ->with(['orderDetails' => function ($query) {
                // 'category', 'brand', 'unit', 'label', 'productImages', 'attributes'
                $query->with(['product' => function ($query) {
                    $query->with(['category', 'brand', 'unit', 'label', 'productImages', 'attributes']);
                }]);
            }])
            ->where('user_id', $user->id);

        if ($request->has('status')) {
            $orders->where('status', $request->status);
        }

        if ($request->has('start_date')) {
            $orders->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $orders->whereDate('created_at', '<=', $request->end_date);
        }

        $perPage = $request->per_page ?? 20;
        $page = $request->page ?? 1;
        $offset = ($page * $perPage) - $perPage;

        $total = $orders->count();

        $orders = $orders->skip($offset)->take($perPage)->get();

        $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $orders,
            $total,
            $perPage,
            $page
        );

        return response()->json([
            'success' => true,
            'message' => 'Orders fetched successfully.',
            'total' => $total,
            'per_page' => $paginator->perPage(),
            'current_page' => $paginator->currentPage(),
            'last_page' => $paginator->lastPage(),
            'orders' => $paginator->values(),
        ]);
    }
    
    public function refund(Request $request)
    {
       $validate = Validator::make($request->all(), [
        //'pending','approved','decline','packing','performa_invoice','invoice_review','delivered','cancelled', 'shipping', 'invoiced' ,'packed','shipped'
            'status' => 'nullable|in:pending,approved,decline,packing,performa_invoice,invoice_review,delivered,cancelled,shipping,invoiced,packed,shipped',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'page' => 'integer',
            'per_page' => 'integer',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $user = auth()->user();
        
        $refundOrderIds = RefundRequest::where('user_id', $user->id)->pluck('order_id')->toArray();

        $orders = Order::with(['shippingAddress', 'billingAddress', 'pickupAddress'])
            ->whereIn('id', $refundOrderIds)
            ->with(['orderDetails' => function ($query) {
                // 'category', 'brand', 'unit', 'label', 'productImages', 'attributes'
                $query->with(['product' => function ($query) {
                    $query->with(['category', 'brand', 'unit', 'label', 'productImages', 'attributes']);
                }]);
            }])
            ->where('user_id', $user->id);

        if ($request->has('status')) {
            $orders->where('status', $request->status);
        }

        if ($request->has('start_date')) {
            $orders->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $orders->whereDate('created_at', '<=', $request->end_date);
        }

        $perPage = $request->per_page ?? 20;
        $page = $request->page ?? 1;
        $offset = ($page * $perPage) - $perPage;

        $total = $orders->count();

        $orders = $orders->skip($offset)->take($perPage)->get();

        $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $orders,
            $total,
            $perPage,
            $page
        );

        return response()->json([
            'success' => true,
            'message' => 'Orders fetched successfully.',
            'total' => $total,
            'per_page' => $paginator->perPage(),
            'current_page' => $paginator->currentPage(),
            'last_page' => $paginator->lastPage(),
            'orders' => $paginator->values(),
        ]);
    }

    public function recent(Request $request)
    {
        $user = auth()->user();

        $orders = $user->orders()->with(['shippingAddress', 'billingAddress', 'pickupAddress'])
            ->with(['orderDetails' => function ($query) {
                // 'category', 'brand', 'unit', 'label', 'productImages', 'attributes'
                $query->with(['product' => function ($query) {
                    $query->with(['category', 'brand', 'unit', 'label', 'productImages', 'attributes']);
                }]);
            }])->latest()->limit(3)->get();

        return response()->json([
            'success' => true,
            'message' => 'Recent orders fetched successfully.',
            'data' => $orders,
        ]);
    }

    public function invoices(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $user = auth()->user();

        $invoices = Invoice::with(['order.orderDetails','company'])
        ->with(['order' => function ($query) {
            $query->with(['shippingAddress', 'billingAddress', 'pickupAddress']);
            $query->with(['orderDetails' => function ($query) {
                // 'category', 'brand', 'unit', 'label', 'productImages', 'attributes'
                $query->with(['product' => function ($query) {
                    $query->with(['category', 'brand', 'unit', 'label', 'productImages', 'attributes']);
                }]);
            }]);
        }])
        ->whereHas('order', function ($query) use ($user) {
            $query->where('user_id', $user->id);
        });

        if ($request->has('start_date')) {
            $invoices->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $invoices->whereDate('created_at', '<=', $request->end_date);
        }

        $invoices = $invoices->latest()->get();

        return response()->json([
            'success' => true,
            'message' => 'Invoices fetched successfully.',
            'data' => $invoices,
        ]);
    }

    public function refundRequest(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'order_id' => 'required|exists:orders,id,user_id,' . auth()->id(),
            'subject' => 'required|string',
            'description' => 'required|string',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $order = Order::find($request->order_id);

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found.',
            ], 404);
        }

        $order_invoice = $order->invoices()->latest()->first();
        if(!$order_invoice || $order_invoice->status == 'draft'){
            return response()->json([
                'success' => false,
                'message' => 'Refund request can be made only after invoice is generated.',
            ], 400);
        }

        if(!$order->hasRefundRequest()){
            return response()->json([
                'success' => false,
                'message' => 'Order does not have any products to refund.',
            ], 400);
        }

        $refundRequest = RefundRequest::where('order_id', $order->id)->first();
        if ($refundRequest)
        {
            return response()->json([
                'success' => false,
                'message' => 'Refund request already exists for this order.',
            ], 400);
        }


        $refundRequest = new RefundRequest();
        $refundRequest->user_id = auth()->user()->id;
        $refundRequest->subject = $request->subject;
        $refundRequest->description = $request->description;
        $refundRequest->status = 'pending';
        $refundRequest->order_id = $order->id;
        $refundRequest->save();

        return response()->json([
            'success' => true,
            'message' => 'Refund request sent successfully.',
            'data' => $refundRequest,
        ]);
    }

    public function refundRequests(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'status' => 'nullable|in:pending,approved,rejected',
            'page' => 'integer',
            'per_page' => 'integer',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $user = auth()->user();

        $refundRequests = RefundRequest::where('user_id', $user->id)->with('order.orderDetails')
        ->with(['order' => function($q){
               $q->with(['orderDetails' => function ($query) {
                // 'category', 'brand', 'unit', 'label', 'productImages', 'attributes'
                $query->with(['product' => function ($query) {
                    $query->with(['category', 'brand', 'unit', 'label', 'productImages', 'attributes']);
                }]);
            }]);
        }]);

        if ($request->has('status')) {
            $refundRequests->where('status', $request->status);
        }

        $perPage = $request->per_page ?? 20;
        $page = $request->page ?? 1;
        $offset = ($page * $perPage) - $perPage;

        $total = $refundRequests->count();

        $refundRequests = $refundRequests->skip($offset)->take($perPage)->get();

        $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $refundRequests,
            $total,
            $perPage,
            $page
        );

        return response()->json([
            'success' => true,
            'message' => 'Refund requests fetched successfully.',
            'total' => $total,
            'per_page' => $paginator->perPage(),
            'current_page' => $paginator->currentPage(),
            'last_page' => $paginator->lastPage(),
            'refund_requests' => $paginator->values(),
        ]);
    }
}
