<?php

namespace App\Http\Controllers\API;

use App\Helpers\ProductsHelper;
use App\Http\Controllers\Controller;
use App\Models\ApiCartItem;
use App\Models\Cart;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class CartController extends Controller
{
    public function index($cart_identifier)
    {
            $cart = Cart::where('cart_identifier', $cart_identifier)->with([
            'cart_items.product.productImages',
            'cart_items.product.category',
            'cart_items.product.brand',
            'cart_items.product.unit',
            'cart_items.product.label',
            'cart_items.product.attributes',
        ])->first();

        if(!$cart){
            return response()->json([
                'success' => false,
                'message' => 'Cart not found.',
            ], 404);
        }


        return response()->json([
            'success' => true,
            'message' => 'Cart found.',
            'data' => [
                'cart' => $cart->only('id', 'cart_identifier','cart_items','total_amount')
            ],
        ]);
    }
    public function store(Request $request)
    {
        $validate = Validator::make($request->all(),[
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|numeric|min:1',
            'cart_identifier' => 'required|string',
            'user_id' => 'nullable|exists:users,id',
            'action' => 'nullable|string|in:+,-'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        try{

            $product = Product::where('id', $request->product_id)
                ->with('wholeSalePrices','userDiscounts')
                ->first();

            if (!$product) {
                return response()->json([
                    'success' => false,
                    'message' => 'Product not found.',
                ], 404);
            }

            $user = User::find($request->user_id);
            if($user && $user->role_id != 2){
                return response()->json([
                    'success' => false,
                    'message' => 'Only customers can add products to cart.',
                ], 422);
            }


            DB::beginTransaction();
            $cart = Cart::where('cart_identifier', $request->cart_identifier)->first();

            if(!$cart){
                $cart = Cart::create([
                    'cart_identifier' => $request->cart_identifier
                ]);
            }

            $cartItem = ApiCartItem::where('cart_id', $cart->id)
                ->where('product_id', $product->id)
                ->first();


            if($cartItem){
                if($request->action == '+'){
                    $cartItem->quantity += $request->quantity;
                }else if($request->action == '-'){
                    $cartItem->quantity -= $request->quantity;
                }else{
                    return response()->json([
                        'success' => false,
                        'message' => 'Action must be + or -.'
                    ],422);
                }

                if($cartItem->quantity < 0){
                    return response()->json([
                        'success' => false,
                        'message' => 'Quantity cannot be less than zero.'
                    ],422);
                }

                if($cartItem->quantity == 0){
                    $cartItem->delete();
                    DB::commit();
                    $cart->total_amount = $this->calculateTotalAmountOfCart($cart);
                    return response()->json([
                        'success' => true,
                        'message' => 'Product removed from cart successfully.',
                        'data' => [
                            'cart' => $cart->only('id', 'cart_identifier','cart_items','total_amount')
                        ],
                    ]);
                }


                $this->validateQuantity($cartItem->quantity, $product);
                $cartItem->price = ProductsHelper::getProductPriceAccordingToQuantity($product, $cartItem->quantity, $user);
                $cartItem->save();
            }else{
                $this->validateQuantity($request->quantity, $product);
                $cartItem = ApiCartItem::create([
                    'cart_id' => $cart->id,
                    'product_id' => $product->id,
                    'quantity' => $request->quantity,
                    'price' => ProductsHelper::getProductPriceAccordingToQuantity($product, $request->quantity, $user)
                ]);
            }


            DB::commit();
            $cart->total_amount = $this->calculateTotalAmountOfCart($cart);
            return response()->json([
                'success' => true,
                'message' => 'Cart updated successfully.',
                'data' => [
                    'cart' => $cart->only('id', 'cart_identifier','cart_items','total_amount')
                ],
            ]);

        }catch(\Exception $e){
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $e->getMessage() . ' ' . $e->getLine(),

            ], 500);
        }
    }

    private function validateQuantity($requestedQuantity, $product)
    {
        if($requestedQuantity < $product->minimum_purchase_quantity){
            return response()->json([
                'success' => false,
                'message' => 'Minimum purchase quantity is '.$product->minimum_purchase_quantity.'.'
            ],422);
        }

        if($product->maximum_purchase_quantity && $requestedQuantity > $product->maximum_purchase_quantity){
            return response()->json([
                'success' => false,
                'message' => 'Maximum purchase quantity is '.$product->maximum_purchase_quantity.'.'
            ],422);
        }

        if($requestedQuantity > $product->quantity){
            return response()->json([
                'success' => false,
                'message' => 'Product quantity is not enough.'
            ],422);
        }

        return true;
    }

    private function calculateTotalAmountOfCart(Cart $cart)
    {
        $cart->load('cart_items');
        $total_amount = 0;
        foreach ($cart->cart_items as $item) {
            $total_amount += $item->price * $item->quantity;
        }

        return $total_amount;
    }
}
