<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        // dd( \App\Helpers\ProductsHelper::getProductPrice(Product::find(11), true));
        $rules = [
            'category' => 'nullable|exists:categories,id',
            'brand' => 'nullable|exists:brands,id',
            'user_id' => 'nullable|exists:users,id',
            'start_price' => 'numeric',
            'end_price' => 'numeric|gte:start_price',
            'page' => 'integer',
            'per_page' => 'integer',

        ];

        $validate = Validator::make($request->all(), $rules);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $products = Product::select('*')->with('category', 'brand', 'unit', 'label','productImages', 'attributes') ->where('status', 1);

        if ($request->has('category')) {
            $products->where('category_id', $request->category);
        }

        if ($request->has('brand')) {
            $products->where('brand_id', $request->brand);
        }

        $products = $products->get();

        $products = $products->map(function ($product) use ($request) {
            $product->unit_price = \App\Helpers\ProductsHelper::getProductPrice($product, true);
            return $product;
        });

        if ($request->has('start_price')) {
            $start_price = $request->start_price;
            $products = $products->filter(function ($product) use ($start_price) {
                return $product->unit_price >= $start_price;
            });
        }

        if ($request->has('end_price')) {
            $end_price = $request->end_price;
            $products = $products->filter(function ($product) use ($end_price) {
                return $product->unit_price <= $end_price;
            });
        }

        $perPage = $request->per_page ?? 20;
        $page = $request->page ?? 1;
        $offset = ($page * $perPage) - $perPage;

        $total = $products->count();

        $products = $products->slice($offset, $perPage);

        $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $products,
            $total,
            $perPage,
            $page
        );

        return response()->json([
            'success' => true,
            'message' => 'Products fetched successfully.',
            'total' => $total,
            'per_page' => $paginator->perPage(),
            'current_page' => $paginator->currentPage(),
            'last_page' => $paginator->lastPage(),
            'products' => $paginator->values(),
        ]);

    }

    public function featured(Request $request)
    {
        $products = Product::where('is_featured', 1)
            ->where('status', 1)
            ->with('category', 'brand', 'unit', 'label', 'productImages', 'attributes');

        if ($request->has('category')) {
            $products->where('category_id', $request->category);
        }

        if ($request->has('brand')) {
            $products->where('brand_id', $request->brand);
        }

        $products = $products->get();

        $products = $products->map(function ($product) use ($request) {
            $product->unit_price = \App\Helpers\ProductsHelper::getProductPrice($product, true);
            return $product;
        });

        if ($request->has('start_price')) {
            $start_price = $request->start_price;
            $products = $products->filter(function ($product) use ($start_price) {
                return $product->unit_price >= $start_price;
            });
        }

        if ($request->has('end_price')) {
            $end_price = $request->end_price;
            $products = $products->filter(function ($product) use ($end_price) {
                return $product->unit_price <= $end_price;
            });
        }

        $perPage = $request->per_page ?? 20;
        $page = $request->page ?? 1;
        $offset = ($page * $perPage) - $perPage;

        $total = $products->count();

        $products = $products->slice($offset, $perPage);

        $paginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $products,
            $total,
            $perPage,
            $page
        );

        return response()->json([
            'success' => true,
            'message' => 'Products fetched successfully.',
            'total' => $total,
            'per_page' => $paginator->perPage(),
            'current_page' => $paginator->currentPage(),
            'last_page' => $paginator->lastPage(),
            'products' => $paginator->values(),
        ]);




    }

    public function search(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'query' => 'required|string',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $query = $request->query('query');

        $products = Product::where('name', 'like', '%' . $query . '%')
            ->orWhere('short_description', 'like', '%' . $query . '%')
            ->orWhere('sku', 'like', '%' . $query . '%')
            ->limit(5) // Adjust the limit as needed
            ->get([
                'id',
                'name',
                'short_description',
                'sku',
                'thumbnail_image'
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Products fetched successfully.',
            'data' => $products,
        ]);
    }


    public function show($id)
    {
        $product = Product::with('category', 'brand', 'unit', 'label', 'productImages', 'attributes')->find($id);

        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found.',
            ], 404);
        }

        $product->unit_price = \App\Helpers\ProductsHelper::getProductPrice($product, true);

        return response()->json([
            'success' => true,
            'message' => 'Product fetched successfully.',
            'data' => $product,
        ]);
    }

    public function categories(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'is_featured' => 'nullable|int|in:0,1',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation Error.',
                'data' => $validate->errors(),
            ], 422);
        }

        $categories = Category::select('id', 'name', 'parent_id', 'is_featured', 'status', 'icon_image', 'order_no')
            ->withCount('descendants as children_count')
            ->with('descendants')
            ->whereNull('parent_id')
            ->orderBy('order_no', 'asc');

        if ($request->has('is_featured')) {
            $categories->where('is_featured', $request->is_featured);
        }

        $categories = $categories->get();

        return response()->json([
            'success' => true,
            'message' => 'Categories fetched successfully.',
            'data' => $categories,
        ]);
    }
}
